/*
 * Decompiled with CFR 0.152.
 */
package com.github.sarxos.webcam;

import com.github.sarxos.webcam.Webcam;
import com.github.sarxos.webcam.WebcamExceptionHandler;
import com.github.sarxos.webcam.WebcamMotionDetectorAlgorithm;
import com.github.sarxos.webcam.WebcamMotionDetectorDefaultAlgorithm;
import com.github.sarxos.webcam.WebcamMotionEvent;
import com.github.sarxos.webcam.WebcamMotionListener;
import java.awt.Point;
import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WebcamMotionDetector {
    private static final Logger LOG = LoggerFactory.getLogger(WebcamMotionDetector.class);
    private static final AtomicInteger NT = new AtomicInteger(0);
    private static final ThreadFactory THREAD_FACTORY = new DetectorThreadFactory();
    public static final int DEFAULT_INTERVAL = 500;
    private final ExecutorService executor = Executors.newFixedThreadPool(2, THREAD_FACTORY);
    private final List<WebcamMotionListener> listeners = new ArrayList<WebcamMotionListener>();
    private final AtomicBoolean running = new AtomicBoolean(false);
    private volatile boolean motion = false;
    private BufferedImage previousOriginal = null;
    private BufferedImage previousFiltered = null;
    private Webcam webcam = null;
    private volatile int interval = 500;
    private volatile int inertia = -1;
    private volatile long lastMotionTimestamp = 0L;
    private final WebcamMotionDetectorAlgorithm algorithm;

    public WebcamMotionDetector(Webcam webcam, WebcamMotionDetectorAlgorithm algorithm, int interval) {
        this.webcam = webcam;
        this.algorithm = algorithm;
        this.setInterval(interval);
    }

    public WebcamMotionDetector(Webcam webcam, int pixelThreshold, double areaThreshold, int interval) {
        this(webcam, new WebcamMotionDetectorDefaultAlgorithm(pixelThreshold, areaThreshold), interval);
    }

    public WebcamMotionDetector(Webcam webcam, int pixelThreshold, double areaThreshold) {
        this(webcam, pixelThreshold, areaThreshold, 500);
    }

    public WebcamMotionDetector(Webcam webcam, int pixelThreshold) {
        this(webcam, pixelThreshold, 0.2);
    }

    public WebcamMotionDetector(Webcam webcam) {
        this(webcam, 25);
    }

    public void start() {
        if (this.running.compareAndSet(false, true)) {
            this.webcam.open();
            this.executor.submit(new Runner());
            this.executor.submit(new Inverter());
        }
    }

    public void stop() {
        if (this.running.compareAndSet(true, false)) {
            this.webcam.close();
            this.executor.shutdownNow();
        }
    }

    protected void detect() {
        if (!this.webcam.isOpen()) {
            this.motion = false;
            return;
        }
        BufferedImage currentOriginal = this.webcam.getImage();
        if (currentOriginal == null) {
            this.motion = false;
            return;
        }
        BufferedImage currentFiltered = this.algorithm.filter(currentOriginal);
        boolean motionDetected = this.algorithm.detect(this.previousFiltered, currentFiltered);
        if (motionDetected) {
            this.motion = true;
            this.lastMotionTimestamp = System.currentTimeMillis();
            this.notifyMotionListeners(currentOriginal);
        }
        this.previousOriginal = currentOriginal;
        this.previousFiltered = currentFiltered;
    }

    private void notifyMotionListeners(BufferedImage currentOriginal) {
        WebcamMotionEvent wme = new WebcamMotionEvent(this, this.previousOriginal, currentOriginal, this.algorithm.getArea(), this.algorithm.getCog(), this.algorithm.getPoints());
        for (WebcamMotionListener l2 : this.listeners) {
            try {
                l2.motionDetected(wme);
            }
            catch (Exception e2) {
                WebcamExceptionHandler.handle(e2);
            }
        }
    }

    public boolean addMotionListener(WebcamMotionListener l2) {
        return this.listeners.add(l2);
    }

    public WebcamMotionListener[] getMotionListeners() {
        return this.listeners.toArray(new WebcamMotionListener[this.listeners.size()]);
    }

    public boolean removeMotionListener(WebcamMotionListener l2) {
        return this.listeners.remove(l2);
    }

    public int getInterval() {
        return this.interval;
    }

    public void setInterval(int interval) {
        if (interval < 100) {
            throw new IllegalArgumentException("Motion check interval cannot be less than 100 ms");
        }
        this.interval = interval;
    }

    public void setPixelThreshold(int threshold) {
        ((WebcamMotionDetectorDefaultAlgorithm)this.algorithm).setPixelThreshold(threshold);
    }

    public void setAreaThreshold(double threshold) {
        ((WebcamMotionDetectorDefaultAlgorithm)this.algorithm).setAreaThreshold(threshold);
    }

    public void setInertia(int inertia) {
        if (inertia < 0) {
            throw new IllegalArgumentException("Inertia time must not be negative!");
        }
        this.inertia = inertia;
    }

    public void clearInertia() {
        this.inertia = -1;
    }

    public Webcam getWebcam() {
        return this.webcam;
    }

    public boolean isMotion() {
        if (!this.running.get()) {
            LOG.warn("Motion cannot be detected when detector is not running!");
        }
        return this.motion;
    }

    public double getMotionArea() {
        return this.algorithm.getArea();
    }

    public Point getMotionCog() {
        Point cog = this.algorithm.getCog();
        if (cog == null) {
            int w = this.webcam.getViewSize().width;
            int h2 = this.webcam.getViewSize().height;
            cog = new Point(w / 2, h2 / 2);
        }
        return cog;
    }

    public WebcamMotionDetectorAlgorithm getDetectorAlgorithm() {
        return this.algorithm;
    }

    public void setMaxMotionPoints(int i2) {
        this.algorithm.setMaxPoints(i2);
    }

    public int getMaxMotionPoints() {
        return this.algorithm.getMaxPoints();
    }

    public void setPointRange(int i2) {
        this.algorithm.setPointRange(i2);
    }

    public int getPointRange() {
        return this.algorithm.getPointRange();
    }

    private class Inverter
    implements Runnable {
        private Inverter() {
        }

        @Override
        public void run() {
            int delay = 0;
            while (WebcamMotionDetector.this.running.get()) {
                try {
                    Thread.sleep(10L);
                }
                catch (InterruptedException e2) {
                    break;
                }
                int n2 = delay = WebcamMotionDetector.this.inertia != -1 ? WebcamMotionDetector.this.inertia : 2 * WebcamMotionDetector.this.interval;
                if (WebcamMotionDetector.this.lastMotionTimestamp + (long)delay >= System.currentTimeMillis()) continue;
                WebcamMotionDetector.this.motion = false;
            }
        }
    }

    private class Runner
    implements Runnable {
        private Runner() {
        }

        @Override
        public void run() {
            WebcamMotionDetector.this.running.set(true);
            while (WebcamMotionDetector.this.running.get() && WebcamMotionDetector.this.webcam.isOpen()) {
                try {
                    WebcamMotionDetector.this.detect();
                    Thread.sleep(WebcamMotionDetector.this.interval);
                }
                catch (InterruptedException e2) {
                    break;
                }
                catch (Exception e3) {
                    WebcamExceptionHandler.handle(e3);
                }
            }
            WebcamMotionDetector.this.running.set(false);
        }
    }

    private static final class DetectorThreadFactory
    implements ThreadFactory {
        private DetectorThreadFactory() {
        }

        @Override
        public Thread newThread(Runnable runnable) {
            Thread t = new Thread(runnable, String.format("motion-detector-%d", NT.incrementAndGet()));
            t.setUncaughtExceptionHandler(WebcamExceptionHandler.getInstance());
            t.setDaemon(true);
            return t;
        }
    }
}

